<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>IPS Hoops Challenge</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Russo+One&family=Roboto:wght@400;700&display=swap');

        body {
            margin: 0;
            overflow: hidden;
            background-color: #1e293b;
            font-family: 'Russo One', sans-serif;
            touch-action: none;
            user-select: none;
        }

        #game-container {
            position: relative;
            width: 100vw;
            height: 100vh;
            background: radial-gradient(circle at center, #334155 0%, #0f172a 100%);
        }

        /* Floor */
        .floor {
            position: absolute;
            bottom: 0;
            width: 100%;
            height: 20%;
            background: #ca8a04; /* Wood color */
            border-top: 5px solid #fff;
            background-image: repeating-linear-gradient(90deg, transparent 0, transparent 40px, rgba(0,0,0,0.1) 40px, rgba(0,0,0,0.1) 42px);
        }

        canvas {
            display: block;
            width: 100%;
            height: 100%;
            position: absolute;
            top: 0;
            left: 0;
            z-index: 1;
        }

        /* UI Overlay */
        .ui-layer {
            position: absolute;
            top: 0; left: 0; width: 100%; height: 100%;
            pointer-events: none;
            z-index: 10;
            display: flex;
            flex-direction: column;
            justify-content: space-between;
        }

        /* Scoreboard */
        #scoreboard {
            margin-top: 20px;
            align-self: center;
            background: #000;
            border: 4px solid #ef4444;
            border-radius: 10px;
            padding: 15px;
            width: 90%;
            max-width: 600px;
            color: #ef4444; /* LED Red */
            text-align: center;
            box-shadow: 0 0 20px rgba(239, 68, 68, 0.5);
            font-family: 'Roboto', sans-serif; /* Readable font for questions */
        }

        .led-text {
            font-family: 'Russo One', sans-serif;
            color: #fca5a5;
            text-transform: uppercase;
        }

        #question-text {
            color: #fff;
            font-size: 1.1rem;
            line-height: 1.4;
            margin-top: 5px;
            font-weight: bold;
        }

        .stats {
            display: flex;
            justify-content: space-between;
            margin-bottom: 5px;
            font-size: 1.2rem;
            color: #fbbf24;
        }

        /* Screens */
        .screen {
            position: absolute;
            top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(0,0,0,0.9);
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            z-index: 50;
            pointer-events: auto;
            text-align: center;
            color: white;
        }

        .btn-start {
            background: #ea580c;
            border: none;
            padding: 15px 40px;
            font-size: 1.5rem;
            color: white;
            border-radius: 50px;
            font-family: 'Russo One', sans-serif;
            cursor: pointer;
            box-shadow: 0 6px 0 #9a3412;
            transition: transform 0.1s;
            margin-top: 20px;
        }

        .btn-start:active {
            transform: translateY(6px);
            box-shadow: none;
        }

        .btn-back {
            color: #888;
            text-decoration: none;
            font-size: 14px;
            font-family: 'Roboto', sans-serif;
            margin-top: 15px;
        }
        .btn-back:hover { color: #ccc; }

        .hidden { display: none !important; }

        /* Floating Text */
        .float-text {
            position: absolute;
            font-size: 2rem;
            font-weight: bold;
            text-shadow: 2px 2px 0 #000;
            pointer-events: none;
            animation: moveUp 1s forwards;
            z-index: 20;
        }

        @keyframes moveUp {
            0% { transform: translateY(0); opacity: 1; }
            100% { transform: translateY(-50px); opacity: 0; }
        }
    </style>
</head>
<body>
<div id="game-container">
    <div class="floor"></div>
    <canvas id="gameCanvas"></canvas>

    <div class="ui-layer">
        <div id="scoreboard">
            <div class="stats">
                <span>SCORE: <span id="score">0</span></span>
                <span>SOAL: <span id="shots">1</span>/5</span>
            </div>
            <div style="border-top: 2px solid #333; margin: 5px 0;"></div>
            <div class="text-gray-400 text-xs uppercase mb-1">Pertanyaan IPS (Kelas 6)</div>
            <div id="question-text">Loading...</div>
        </div>
        <div class="text-center text-white/50 pb-4 text-sm">Tarik bola ke bawah, arahkan, lalu lepas!</div>
    </div>

    <!-- Start Screen -->
    <div id="start-screen" class="screen">
        <h1 class="text-6xl text-orange-500 mb-2" style="text-shadow: 0 0 20px orange;">IPS HOOPS</h1>
        <p class="text-xl mb-8 text-gray-300">Basket Edukasi Kelas 6 SD</p>
        <div class="bg-gray-800 p-6 rounded-lg max-w-sm text-left text-sm mb-6">
            <p>🏀 <b>Cara Main:</b></p>
            <ul class="list-disc pl-5 mt-2 space-y-2">
                <li>Baca soal IPS di papan skor.</li>
                <li>Lihat jawaban di atas Ring Kiri & Kanan.</li>
                <li><b>Tarik bola</b> seperti ketapel untuk membidik.</li>
                <li>Jika <b>SALAH</b>, bola akan kembali dan kamu bisa coba lagi!</li>
            </ul>
        </div>
        <button class="btn-start" onclick="initGame()">TIP OFF!</button>
        <a href="index.php?game_id=7&spawn=entrance" class="btn-back">← Kembali ke Lobby</a>
    </div>

    <!-- End Screen -->
    <div id="end-screen" class="screen hidden">
        <h1 class="text-5xl text-white mb-4">GAME OVER</h1>
        <div class="text-2xl mb-2 text-yellow-400">Score Akhir: <span id="final-score">0</span></div>
        <div class="text-gray-400 mb-8" id="end-msg">Good Game!</div>
        <button class="btn-start" onclick="location.reload()">MAIN LAGI</button>
        <a href="index.php?game_id=7&spawn=entrance" class="btn-back">← Kembali ke Lobby</a>
    </div>
</div>

<script>
    const canvas = document.getElementById('gameCanvas');
    const ctx = canvas.getContext('2d');

    // --- CONFIG ---
    let width, height;
    function resize() {
        width = window.innerWidth;
        height = window.innerHeight;
        canvas.width = width;
        canvas.height = height;
    }
    window.addEventListener('resize', resize);
    resize();

    // --- DATA SOAL IPS ---
    let questions = [
        {
            q: "Kapan organisasi ASEAN didirikan?",
            a: "8 Ags 1967",
            b: "17 Ags 1945",
            correct: 'left'
        },
        {
            q: "Siapa tokoh yang mengetik naskah Proklamasi?",
            a: "Sayuti Melik",
            b: "Soekarno",
            correct: 'left'
        },
        {
            q: "Kegiatan mengirim barang ke luar negeri disebut...",
            a: "Impor",
            b: "Ekspor",
            correct: 'right'
        },
        {
            q: "Negara tetangga yang berbatasan darat dengan Kalimantan adalah...",
            a: "Malaysia",
            b: "Singapura",
            correct: 'left'
        },
        {
            q: "Perjanjian Linggarjati ditandatangani pada tahun...",
            a: "1955",
            b: "1946",
            correct: 'right'
        }
    ];

    // Fetch Questions from DB
    async function fetchQuestions() {
        const urlParams = new URLSearchParams(window.location.search);
        const gameId = urlParams.get('game_id') || 7;
        try {
            const res = await fetch(`questions.php?game_id=${gameId}`);
            if (res.ok) {
                const data = await res.json();
                if (data && Object.keys(data).length > 0) {
                    const newQuestions = [];
                    Object.keys(data).forEach(lvl => {
                        const q = data[lvl];
                        const correctAns = q.options[q.correct];
                        const wrongAns = q.options.filter((_, i) => i !== q.correct)[0] || "Salah";
                        
                        // Randomize position logic handled here for data structure, 
                        // but game also has visual randomization. 
                        // Let's stick to the data structure expected: 'left' means 'a' is correct, 'right' means 'b' is correct.
                        
                        const isLeftCorrect = Math.random() > 0.5;
                        
                        newQuestions.push({
                            q: q.q,
                            a: isLeftCorrect ? correctAns : wrongAns,
                            b: isLeftCorrect ? wrongAns : correctAns,
                            correct: isLeftCorrect ? 'left' : 'right'
                        });
                    });
                    
                    if (newQuestions.length > 0) {
                        while (newQuestions.length < 5) {
                            newQuestions.push(newQuestions[newQuestions.length % newQuestions.length]);
                        }
                        questions = newQuestions;
                        console.log("Questions loaded from DB:", questions);
                    }
                }
            }
        } catch (e) {
            console.error("Failed to fetch questions:", e);
        }
    }

    // Init Data
    fetchQuestions();

    // --- GAME STATE ---
    let gameState = 'MENU';
    let currentQ = 0;
    let score = 0;
    let ball = { x: 0, y: 0, prevX: 0, prevY: 0, vx: 0, vy: 0, radius: 25, active: false, dragging: false };
    let dragStart = { x: 0, y: 0 };
    let dragCurrent = { x: 0, y: 0 };
    let hoops = [];
    let gravity = 0.5;
    let bounceFactor = -0.6;
    let shotMade = false;
    let waitingNext = false;
    let predictionPoints = [];
    let throwStartTime = 0; // Variabel baru untuk mencegah stuck

    // --- SETUP ---
    function initGame() {
        document.getElementById('start-screen').classList.add('hidden');
        document.getElementById('end-screen').classList.add('hidden');
        score = 0;
        currentQ = 0;
        
        resetLevel();
        gameState = 'PLAY';
        gameLoop();
    }

    function resetLevel() {
        if (currentQ >= questions.length) {
            endGame();
            return;
        }

        shotMade = false;
        waitingNext = false;
        updateUI();

        // Reset Ball Position (Bottom Center)
        resetBall();

        // Setup Hoops (Left and Right)
        const qData = questions[currentQ];
        const isLeftCorrect = Math.random() > 0.5; // Randomize visual position
        
        hoops = [
            {
                x: width * 0.25,
                y: height * 0.4,
                w: 90,
                label: isLeftCorrect ? qData.a : qData.b,
                isCorrect: false
            },
            {
                x: width * 0.75,
                y: height * 0.4,
                w: 90,
                label: '',
                isCorrect: false
            }
        ];

        // Assign labels logic
        const leftIsA = Math.random() > 0.5;
        
        hoops[0].label = leftIsA ? questions[currentQ].a : questions[currentQ].b;
        hoops[1].label = leftIsA ? questions[currentQ].b : questions[currentQ].a;

        const correctAns = questions[currentQ].correct === 'left' ? questions[currentQ].a : questions[currentQ].b; 
        
        hoops[0].isCorrect = hoops[0].label === correctAns;
        hoops[1].isCorrect = hoops[1].label === correctAns;
    }

    function resetBall() {
        ball.x = width / 2;
        ball.y = height - 120; // Slightly higher to give room for dragging down
        ball.prevX = ball.x;
        ball.prevY = ball.y;
        ball.vx = 0;
        ball.vy = 0;
        ball.active = false;
        ball.dragging = false;
        waitingNext = false;
        shotMade = false;
    }

    function updateUI() {
        if (currentQ < questions.length) {
            document.getElementById('question-text').innerText = questions[currentQ].q;
            document.getElementById('score').innerText = score;
            document.getElementById('shots').innerText = (currentQ + 1);
        }
    }

    function endGame() {
        gameState = 'END';
        document.getElementById('end-screen').classList.remove('hidden');
        document.getElementById('final-score').innerText = score;
        
        const msg = document.getElementById('end-msg');
        if(score >= 400) msg.innerText = "Luar Biasa! Nilai Sempurna!";
        else if(score >= 200) msg.innerText = "Bagus! Coba lagi untuk nilai sempurna.";
        else msg.innerText = "Terus belajar dan coba lagi!";
    }

    // --- INPUT HANDLERS (Improved) ---
    // Using window for move/up ensures we don't lose drag if mouse leaves canvas
    canvas.addEventListener('mousedown', startDrag);
    window.addEventListener('mousemove', moveDrag);
    window.addEventListener('mouseup', endDrag);
    
    canvas.addEventListener('touchstart', (e) => startDrag(e.touches[0]));
    window.addEventListener('touchmove', (e) => { 
        // Only prevent default if we are dragging the ball
        if(ball.dragging) e.preventDefault(); 
        if(e.touches.length > 0) moveDrag(e.touches[0]); 
    }, {passive: false});
    window.addEventListener('touchend', endDrag);

    function startDrag(e) {
        if (ball.active || waitingNext) return;
        
        // Check if touching ball
        const dist = Math.hypot(e.clientX - ball.x, e.clientY - ball.y);
        if (dist < ball.radius * 4) { // Large hit area for easy grabbing
            ball.dragging = true;
            dragStart = { x: ball.x, y: ball.y }; 
            dragCurrent = { x: e.clientX, y: e.clientY };
            updatePrediction();
        }
    }

    function moveDrag(e) {
        if (ball.dragging) {
            dragCurrent = { x: e.clientX, y: e.clientY };
            updatePrediction();
        }
    }

    function endDrag() {
        if (ball.dragging) {
            ball.dragging = false;
            ball.active = true;
            
            // Power multiplier
            const power = 0.18; 
            
            // Calculate velocity
            let vx = (dragStart.x - dragCurrent.x) * power;
            let vy = (dragStart.y - dragCurrent.y) * power;
            
            // Removed strict hard cap that caused "stuck" feeling
            // Only ensure it doesn't shoot downwards instantly
            if (vy > 0) vy = -2; // Minimal upward pop if dragged upwards

            ball.vx = vx;
            ball.vy = vy;
            
            predictionPoints = []; 
            throwStartTime = Date.now(); // Catat waktu lemparan
        }
    }

    function updatePrediction() {
        predictionPoints = [];
        const power = 0.18;
        let simX = ball.x;
        let simY = ball.y;
        let simVx = (dragStart.x - dragCurrent.x) * power;
        let simVy = (dragStart.y - dragCurrent.y) * power;
        
        if (simVy > 0) simVy = -2;

        for(let i=0; i<35; i++) { // Predict slightly longer trajectory
            simX += simVx;
            simY += simVy;
            simVy += gravity;
            if(i % 3 === 0) predictionPoints.push({x: simX, y: simY});
        }
    }

    // --- PHYSICS & LOGIC ---
    function update() {
        if (ball.active && !waitingNext) {
            // Failsafe: Jika bola melayang/memantul lebih dari 5 detik, paksa reset
            if (Date.now() - throwStartTime > 5000) {
                handleMiss();
                return;
            }

            ball.prevX = ball.x;
            ball.prevY = ball.y;
            
            ball.x += ball.vx;
            ball.y += ball.vy;
            ball.vy += gravity;

            // Wall bounce
            if (ball.x - ball.radius < 0 || ball.x + ball.radius > width) {
                ball.vx *= bounceFactor;
                ball.x = ball.x < width/2 ? ball.radius : width - ball.radius;
            }

            // Floor bounce
            if (ball.y + ball.radius > height) {
                ball.vy *= bounceFactor;
                ball.y = height - ball.radius;
                ball.vx *= 0.90; // Friction diperbesar sedikit agar cepat berhenti

                // If stopped moving or slow enough at bottom (Kondisi diperlonggar)
                if (Math.abs(ball.vy) < 2 && Math.abs(ball.vx) < 0.5) {
                     handleMiss();
                }
            }

            // Hoop Collision
            hoops.forEach(hoop => {
                if (ball.prevY < hoop.y && ball.y >= hoop.y) {
                    const fraction = (hoop.y - ball.prevY) / (ball.y - ball.prevY);
                    const intersectX = ball.prevX + (ball.x - ball.prevX) * fraction;

                    const tolerance = 15; // More forgiving tolerance
                    if (intersectX > hoop.x - hoop.w/2 - tolerance && intersectX < hoop.x + hoop.w/2 + tolerance) {
                        handleScore(hoop);
                    }
                }
            });
            
            // Out of bounds (bottom)
            if (ball.y > height + 50) handleMiss();
        }
    }

    function handleScore(hoop) {
        if (shotMade) return;
        shotMade = true;
        waitingNext = true;

        if (hoop.isCorrect) {
            // Correct!
            score += 100;
            createFloatText(hoop.x, hoop.y, "BENAR! +100", "#4ade80");
            
            updateUI();
            
            // Move to next question
            setTimeout(() => {
                currentQ++;
                resetLevel();
            }, 1500);
        } else {
            // Wrong hoop - RETRY LOGIC
            createFloatText(hoop.x, hoop.y, "SALAH! Coba Lagi", "#ef4444");
            
            // Do NOT increment score, Do NOT increment currentQ
            // Just reset ball to try again
            setTimeout(() => {
                resetBall();
            }, 1500);
        }
    }

    function handleMiss() {
        if (shotMade || waitingNext) return;
        waitingNext = true;
        
        createFloatText(ball.x, ball.y, "MISSED!", "#ffffff");
        
        setTimeout(() => {
            resetBall();
        }, 1000);
    }

    function createFloatText(x, y, text, color) {
        const el = document.createElement('div');
        el.className = 'float-text';
        el.style.left = x + 'px';
        el.style.top = y + 'px';
        el.innerText = text;
        el.style.color = color;
        // Adjust position to center text
        el.style.transform = 'translate(-50%, -50%)';
        document.body.appendChild(el);
        setTimeout(() => el.remove(), 1000);
    }

    // --- RENDERING ---
    function draw() {
        ctx.clearRect(0, 0, width, height);

        // Draw Hoops
        hoops.forEach(hoop => {
            // Backboard
            ctx.fillStyle = hoop.isCorrect && waitingNext && shotMade ? '#22c55e' : '#fff'; 
            ctx.fillRect(hoop.x - 50, hoop.y - 60, 100, 60);
            ctx.strokeStyle = '#999';
            ctx.lineWidth = 2;
            ctx.strokeRect(hoop.x - 50, hoop.y - 60, 100, 60);

            // Text on Backboard
            ctx.fillStyle = '#000';
            ctx.font = 'bold 14px Roboto';
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            ctx.fillText(hoop.label, hoop.x, hoop.y - 30);

            // Rim
            ctx.beginPath();
            ctx.moveTo(hoop.x - hoop.w/2, hoop.y);
            ctx.lineTo(hoop.x + hoop.w/2, hoop.y);
            ctx.strokeStyle = '#ea580c';
            ctx.lineWidth = 5;
            ctx.stroke();
            
            // Net (Visual)
            ctx.beginPath();
            ctx.moveTo(hoop.x - hoop.w/2, hoop.y);
            ctx.lineTo(hoop.x - hoop.w/3, hoop.y + 40);
            ctx.lineTo(hoop.x + hoop.w/3, hoop.y + 40);
            ctx.lineTo(hoop.x + hoop.w/2, hoop.y);
            ctx.strokeStyle = 'rgba(255,255,255,0.5)';
            ctx.lineWidth = 2;
            ctx.stroke();
        });

        // Draw Prediction Line (Dots)
        if (ball.dragging && predictionPoints.length > 0) {
            ctx.fillStyle = 'rgba(255, 255, 255, 0.7)';
            predictionPoints.forEach(p => {
                ctx.beginPath();
                ctx.arc(p.x, p.y, 4, 0, Math.PI * 2);
                ctx.fill();
            });
            
            // Draw Rubber Band
            ctx.beginPath();
            ctx.moveTo(ball.x, ball.y);
            ctx.lineTo(dragCurrent.x, dragCurrent.y);
            ctx.strokeStyle = 'rgba(255, 255, 255, 0.3)';
            ctx.lineWidth = 2;
            ctx.stroke();
        }

        // Draw Ball
        ctx.save();
        ctx.translate(ball.x, ball.y);
        
        ctx.beginPath();
        ctx.arc(0, 0, ball.radius, 0, Math.PI * 2);
        ctx.fillStyle = '#f97316';
        ctx.fill();
        ctx.strokeStyle = '#000';
        ctx.lineWidth = 2;
        ctx.stroke();

        ctx.beginPath();
        ctx.moveTo(0, -ball.radius);
        ctx.bezierCurveTo(ball.radius/2, -ball.radius/2, ball.radius/2, ball.radius/2, 0, ball.radius);
        ctx.stroke();
        
        ctx.beginPath();
        ctx.moveTo(-ball.radius, 0);
        ctx.lineTo(ball.radius, 0);
        ctx.stroke();

        ctx.restore();
    }

    function gameLoop() {
        update();
        draw();
        if (gameState !== 'MENU') requestAnimationFrame(gameLoop);
    }

</script>
</body>
</html>


