<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Neon Dash: Math Adventure</title>
    <style>
        body {
            margin: 0;
            padding: 0;
            background-color: #050505;
            overflow: hidden;
            font-family: 'Courier New', Courier, monospace;
            touch-action: none;
        }

        #gameCanvas {
            display: block;
            margin: 0 auto;
            background: radial-gradient(circle at center, #1a1a2e 0%, #000000 100%);
        }

        #ui-layer {
            position: absolute;
            top: 20px;
            left: 20px;
            color: #fff;
            font-size: 20px;
            pointer-events: none;
            text-shadow: 0 0 10px #0ff;
            z-index: 5;
        }

        .btn-back {
            position: absolute;
            top: 20px;
            right: 20px;
            background: transparent;
            color: #0ff;
            border: 2px solid #0ff;
            padding: 8px 16px;
            font-size: 14px;
            cursor: pointer;
            font-family: inherit;
            text-transform: uppercase;
            transition: all 0.2s;
            box-shadow: 0 0 10px #0ff;
            text-decoration: none;
            z-index: 10;
        }

        .btn-back:hover {
            background: #0ff;
            color: #000;
        }

        /* Modal Styles */
        #start-screen, #game-over-screen, #quiz-modal {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            text-align: center;
            color: white;
            background: rgba(0, 0, 0, 0.9);
            padding: 30px;
            border: 2px solid #0ff;
            border-radius: 10px;
            box-shadow: 0 0 20px #0ff;
            display: flex;
            flex-direction: column;
            gap: 15px;
            z-index: 20;
            min-width: 300px;
        }

        h1 {
            margin: 0;
            font-size: 32px;
            color: #0ff;
            text-transform: uppercase;
            letter-spacing: 3px;
            text-shadow: 0 0 20px #0ff;
        }

        h2 {
            margin: 0;
            font-size: 24px;
            color: #ff0;
        }

        p {
            font-size: 16px;
            color: #ddd;
            line-height: 1.5;
        }

        button {
            background: transparent;
            color: #0ff;
            border: 2px solid #0ff;
            padding: 10px 20px;
            font-size: 18px;
            cursor: pointer;
            font-family: inherit;
            text-transform: uppercase;
            transition: all 0.2s;
            box-shadow: 0 0 10px #0ff;
            margin-top: 10px;
        }

        button:hover {
            background: #0ff;
            color: #000;
            box-shadow: 0 0 20px #0ff;
        }

        .quiz-option {
            background: rgba(255, 255, 255, 0.1);
            border: 1px solid #aaa;
            padding: 10px;
            margin: 5px 0;
            cursor: pointer;
            transition: background 0.2s;
            color: #fff;
            font-size: 18px;
            display: block;
            width: 100%;
            box-sizing: border-box;
        }

        .quiz-option:hover {
            background: rgba(0, 255, 255, 0.3);
            border-color: #0ff;
        }

        .hidden {
            display: none !important;
        }

        /* Mobile Controls */
        #mobile-controls {
            position: absolute;
            bottom: 20px;
            width: 100%;
            height: 120px;
            display: none;
            pointer-events: none;
            z-index: 10;
        }

        .control-btn {
            position: absolute;
            width: 60px;
            height: 60px;
            background: rgba(255, 255, 255, 0.1);
            border: 2px solid rgba(0, 255, 255, 0.3);
            border-radius: 50%;
            pointer-events: auto;
            display: flex;
            align-items: center;
            justify-content: center;
            color: rgba(255, 255, 255, 0.5);
            font-size: 24px;
            user-select: none;
        }

        .control-btn:active {
            background: rgba(0, 255, 255, 0.3);
        }

        #btn-left { left: 20px; bottom: 20px; }
        #btn-right { left: 90px; bottom: 20px; }
        #btn-jump { right: 20px; bottom: 20px; border-color: #f0f; color: #f0f; }
        #btn-dash { right: 90px; bottom: 20px; border-color: #ff0; color: #ff0; }

    </style>
</head>
<body>
<!-- Back Button -->
    <a href="index.php?game_id=3&spawn=entrance" class="btn-back">KEMBALI</a>

    <div id="ui-layer">
        SOAL: <span id="score">0</span>/6 | MATI: <span id="deaths">0</span>
    </div>

    <!-- Start Screen -->
    <div id="start-screen">
        <h1>NEON DASH</h1>
        <p>Jawab 6 soal matematika kelas 6 SD untuk menang!<br>Gunakan Arrow Keys/WASD untuk gerak.<br>SPACE untuk Lompat, SHIFT untuk Dash.</p>
        <button onclick="startGame()">Mulai Petualangan</button>
    </div>

    <!-- Quiz Modal -->
    <div id="quiz-modal" class="hidden">
        <h2 id="quiz-question">Soal Matematika</h2>
        <div id="quiz-options">
            <!-- Options injected by JS -->
        </div>
        <p id="quiz-feedback" style="color: #f00; display: none;">Jawaban Salah, Coba Lagi!</p>
    </div>

    <!-- Game Over Screen -->
    <div id="game-over-screen" class="hidden">
        <h1 id="go-title">LEVEL SELESAI</h1>
        <p id="go-msg">Kamu berhasil menyelesaikan semua soal!</p>
        <button onclick="resetLevel()">Main Lagi</button>
    </div>

    <div id="mobile-controls">
        <div id="btn-left" class="control-btn">←</div>
        <div id="btn-right" class="control-btn">→</div>
        <div id="btn-dash" class="control-btn">⚡</div>
        <div id="btn-jump" class="control-btn">▲</div>
    </div>

    <canvas id="gameCanvas"></canvas>

<script>
    const canvas = document.getElementById('gameCanvas');
    const ctx = canvas.getContext('2d');
    const scoreEl = document.getElementById('score');
    const deathsEl = document.getElementById('deaths');
    
    // Screens
    const startScreen = document.getElementById('start-screen');
    const gameOverScreen = document.getElementById('game-over-screen');
    const quizModal = document.getElementById('quiz-modal');
    const quizQuestionEl = document.getElementById('quiz-question');
    const quizOptionsEl = document.getElementById('quiz-options');
    const quizFeedbackEl = document.getElementById('quiz-feedback');
    const goTitle = document.getElementById('go-title');
    const goMsg = document.getElementById('go-msg');

    // Game State
    let gameRunning = false;
    let isPaused = false; // Untuk pause saat kuis muncul
    let score = 0;
    let deaths = 0;
    let lastTime = 0;
    
    // Data Soal Default (Fallback)
    let mathQuestions = [
        { q: "FPB dari 12 dan 18 adalah?", options: ["6", "3", "12"], ans: 0 },
        { q: "50% dari 200 adalah?", options: ["50", "100", "20"], ans: 1 },
        { q: "Hasil dari -5 + 12 adalah?", options: ["-7", "7", "17"], ans: 1 },
        { q: "Volume kubus dengan sisi 3 cm?", options: ["9 cm³", "12 cm³", "27 cm³"], ans: 2 },
        { q: "2 jam + 30 menit = ... menit?", options: ["120", "150", "130"], ans: 1 },
        { q: "0,75 jika diubah ke pecahan biasa?", options: ["1/2", "3/4", "2/3"], ans: 1 }
    ];

    // Fetch Questions from DB
    async function fetchQuestions() {
        const urlParams = new URLSearchParams(window.location.search);
        const gameId = urlParams.get('game_id') || 3;
        try {
            const res = await fetch(`questions.php?game_id=${gameId}`);
            if (res.ok) {
                const data = await res.json();
                if (data && Object.keys(data).length > 0) {
                    // Map DB data to game format
                    // DB: { "1": { q, options, correct }, "2": ... }
                    // Game: [ { q, options, ans }, ... ]
                    
                    const newQuestions = [];
                    Object.keys(data).forEach(lvl => {
                        const q = data[lvl];
                        newQuestions.push({
                            q: q.q,
                            options: q.options,
                            ans: q.correct
                        });
                    });
                    
                    // If we have questions, override the default
                    if (newQuestions.length > 0) {
                        // Repeat questions if fewer than 6 (since game expects 6 coins)
                        while (newQuestions.length < 6) {
                            newQuestions.push(newQuestions[newQuestions.length % newQuestions.length]);
                        }
                        mathQuestions = newQuestions;
                        console.log("Questions loaded from DB:", mathQuestions);
                    }
                }
            }
        } catch (e) {
            console.error("Failed to fetch questions:", e);
        }
    }

    // Init Data
    fetchQuestions();
    let currentPendingCoinIndex = -1; // Koin mana yang sedang diambil

    // Input State
    const keys = { right: false, left: false, up: false, dash: false };

    // Camera
    const camera = { x: 0, y: 0 };

    // Physics Constants
    const GRAVITY = 0.6;
    const FRICTION = 0.8;
    const SPEED = 5;
    const JUMP_FORCE = 12;
    const DASH_SPEED = 15;

    // Mobile Check
    const isTouchDevice = 'ontouchstart' in window || navigator.maxTouchPoints > 0;
    if (isTouchDevice) {
        document.getElementById('mobile-controls').style.display = 'block';
    }

    class Particle {
        constructor(x, y, color, speed, size) {
            this.x = x;
            this.y = y;
            this.color = color;
            this.size = size || Math.random() * 3 + 1;
            this.vx = (Math.random() - 0.5) * speed;
            this.vy = (Math.random() - 0.5) * speed;
            this.life = 1.0;
            this.decay = Math.random() * 0.03 + 0.01;
        }
        update() {
            this.x += this.vx;
            this.y += this.vy;
            this.life -= this.decay;
        }
        draw(ctx) {
            ctx.globalAlpha = this.life;
            ctx.fillStyle = this.color;
            ctx.shadowBlur = 10;
            ctx.shadowColor = this.color;
            ctx.fillRect(this.x, this.y, this.size, this.size);
            ctx.globalAlpha = 1.0;
            ctx.shadowBlur = 0;
        }
    }

    class Player {
        constructor(x, y) {
            this.startPos = {x, y};
            this.x = x;
            this.y = y;
            this.w = 30;
            this.h = 30;
            this.vx = 0;
            this.vy = 0;
            this.color = '#0ff';
            this.grounded = false;
            this.jumps = 0;
            this.maxJumps = 2;
            this.isDashing = false;
            this.dashCooldown = 0;
            this.dashTimer = 0;
            this.facingRight = true;
        }

        reset() {
            // Particle effect death
            for(let i=0; i<20; i++) {
                particles.push(new Particle(this.x + this.w/2, this.y + this.h/2, '#f00', 10));
            }
            this.x = this.startPos.x;
            this.y = this.startPos.y;
            this.vx = 0;
            this.vy = 0;
            deaths++;
            deathsEl.innerText = deaths;
        }

        update() {
            if(isPaused) return; // Stop movement when quiz open

            // Movement Logic
            if (keys.right) {
                this.vx += 1;
                this.facingRight = true;
            } else if (keys.left) {
                this.vx -= 1;
                this.facingRight = false;
            } else {
                this.vx *= FRICTION;
            }

            // Dash
            if (keys.dash && this.dashCooldown <= 0 && (keys.right || keys.left)) {
                this.isDashing = true;
                this.dashTimer = 10;
                this.dashCooldown = 60;
                this.vy = 0;
                this.vx = this.facingRight ? DASH_SPEED : -DASH_SPEED;
                for(let i=0; i<10; i++) {
                    particles.push(new Particle(this.x + this.w/2, this.y + this.h/2, '#fff', 5));
                }
            }

            if (this.dashTimer > 0) {
                this.dashTimer--;
                this.vy = 0;
            } else {
                this.isDashing = false;
                if (this.vx > SPEED) this.vx = SPEED;
                if (this.vx < -SPEED) this.vx = -SPEED;
                this.vy += GRAVITY;
            }
            if (this.dashCooldown > 0) this.dashCooldown--;

            this.x += this.vx;
            this.y += this.vy;

            // Trail
            if (!isPaused && (Math.abs(this.vx) > 3 || Math.abs(this.vy) > 3)) {
                if (Math.random() > 0.5) {
                    particles.push(new Particle(this.x + this.w/2, this.y + this.h/2, this.color, 1));
                }
            }

            this.checkCollisions();

            // Fall death
            if (this.y > canvas.height + 200) {
                this.reset();
            }
        }

        jump() {
            if (isPaused) return;
            if (this.jumps < this.maxJumps) {
                this.vy = -JUMP_FORCE;
                this.jumps++;
                this.grounded = false;
                for(let i=0; i<5; i++) {
                    particles.push(new Particle(this.x + this.w/2, this.y + this.h, '#fff', 3));
                }
            }
        }

        checkCollisions() {
            this.grounded = false;

            // Platform Collision
            for (let platform of platforms) {
                if (this.x < platform.x + platform.w &&
                    this.x + this.w > platform.x &&
                    this.y < platform.y + platform.h &&
                    this.y + this.h > platform.y) {
                    
                    if (platform.type === 'danger') {
                        this.reset();
                        return;
                    }

                    let dx = (this.x + this.w / 2) - (platform.x + platform.w / 2);
                    let dy = (this.y + this.h / 2) - (platform.y + platform.h / 2);
                    let width = (this.w + platform.w) / 2;
                    let height = (this.h + platform.h) / 2;
                    let crossWidth = width * dy;
                    let crossHeight = height * dx;

                    if (Math.abs(dx) <= width && Math.abs(dy) <= height) {
                        if (crossWidth > crossHeight) {
                            if (crossWidth > -crossHeight) {
                                this.y = platform.y + platform.h;
                                this.vy = 0;
                            } else {
                                this.x = platform.x - this.w;
                                this.vx = 0;
                            }
                        } else {
                            if (crossWidth > -crossHeight) {
                                this.x = platform.x + platform.w;
                                this.vx = 0;
                            } else {
                                this.y = platform.y - this.h;
                                this.vy = 0;
                                this.grounded = true;
                                this.jumps = 0;
                            }
                        }
                    }
                }
            }

            // Coin Collision (Trigger Quiz)
            for (let i = 0; i < coins.length; i++) {
                let c = coins[i];
                if (!c.collected && 
                    this.x < c.x + c.size &&
                    this.x + this.w > c.x &&
                    this.y < c.y + c.size &&
                    this.y + this.h > c.y) {
                    
                    // Trigger Quiz
                    triggerQuiz(i);
                }
            }

            // Portal Collision
            if (this.x < portal.x + portal.w &&
                this.x + this.w > portal.x &&
                this.y < portal.y + portal.h &&
                this.y + this.h > portal.y) {
                
                if (score >= 6) {
                    winGame();
                } else {
                    // Feedback visual jika belum semua koin diambil
                    ctx.fillStyle = '#fff';
                    ctx.fillText("Kumpulkan semua soal!", this.x - 50, this.y - 20);
                }
            }
        }

        draw(ctx) {
            ctx.fillStyle = this.isDashing ? '#fff' : this.color;
            ctx.shadowBlur = 15;
            ctx.shadowColor = this.color;
            ctx.fillRect(this.x, this.y, this.w, this.h);

            // Eyes
            ctx.fillStyle = '#000';
            ctx.shadowBlur = 0;
            if (this.facingRight) {
                ctx.fillRect(this.x + 18, this.y + 8, 8, 8);
            } else {
                ctx.fillRect(this.x + 4, this.y + 8, 8, 8);
            }
        }
    }

    let player;
    let platforms = [];
    let coins = [];
    let particles = [];
    let portal = {};

    function initLevel() {
        platforms = [];
        coins = [];
        score = 0;
        scoreEl.innerText = "0";
        
        player = new Player(100, 400);

        // --- LEVEL DESIGN EXTENDED (Target ~5000px length) ---

        // SECTION 1: Intro (0 - 1000px)
        platforms.push({x: 0, y: 500, w: 800, h: 50, type: 'normal'});
        // Coin 1 (Easy jump)
        platforms.push({x: 500, y: 350, w: 100, h: 20, type: 'normal'});
        coins.push({x: 540, y: 310, size: 20, collected: false, qIndex: 0});

        // SECTION 2: The Lava Pit (1000 - 2000px)
        platforms.push({x: 900, y: 500, w: 200, h: 20, type: 'normal'});
        platforms.push({x: 1200, y: 550, w: 600, h: 50, type: 'danger'}); // Lava
        platforms.push({x: 1250, y: 400, w: 80, h: 20, type: 'normal'}); // Island 1
        platforms.push({x: 1450, y: 350, w: 80, h: 20, type: 'normal'}); // Island 2
        // Coin 2 (Above lava)
        coins.push({x: 1480, y: 300, size: 20, collected: false, qIndex: 1});
        platforms.push({x: 1650, y: 300, w: 150, h: 20, type: 'normal'}); // Safe landing

        // SECTION 3: The Climb (2000 - 3000px)
        platforms.push({x: 1900, y: 500, w: 400, h: 20, type: 'normal'}); // Floor
        platforms.push({x: 2100, y: 400, w: 50, h: 20, type: 'normal'}); // Step 1
        platforms.push({x: 2200, y: 300, w: 50, h: 20, type: 'normal'}); // Step 2
        platforms.push({x: 2100, y: 200, w: 50, h: 20, type: 'normal'}); // Step 3
        platforms.push({x: 2300, y: 150, w: 200, h: 20, type: 'normal'}); // High platform
        // Coin 3 (High up)
        coins.push({x: 2380, y: 100, size: 20, collected: false, qIndex: 2});

        // SECTION 4: The Dash Test (3000 - 4000px)
        // Must dash through gaps
        platforms.push({x: 2600, y: 400, w: 1000, h: 50, type: 'danger'}); // Long Lava
        platforms.push({x: 2600, y: 250, w: 200, h: 20, type: 'normal'}); // Start Dash
        platforms.push({x: 3000, y: 250, w: 200, h: 20, type: 'normal'}); // Land Dash
        // Coin 4 (Between dash)
        coins.push({x: 2900, y: 200, size: 20, collected: false, qIndex: 3});
        platforms.push({x: 3400, y: 250, w: 200, h: 20, type: 'normal'});

        // SECTION 5: Tricky Steps (4000 - 5000px)
        platforms.push({x: 3700, y: 350, w: 50, h: 100, type: 'normal'}); // Pillar
        platforms.push({x: 3900, y: 300, w: 50, h: 150, type: 'normal'}); // Pillar
        // Coin 5 (On pillar)
        coins.push({x: 3915, y: 250, size: 20, collected: false, qIndex: 4});
        platforms.push({x: 4100, y: 400, w: 50, h: 50, type: 'normal'}); // Small step

        // SECTION 6: Final Stretch (5000px+)
        platforms.push({x: 4300, y: 500, w: 800, h: 50, type: 'normal'}); // Final floor
        // Coin 6 (Before portal)
        coins.push({x: 4500, y: 450, size: 20, collected: false, qIndex: 5});

        // Portal at the end
        portal = {x: 4800, y: 400, w: 60, h: 100};
    }

    // --- QUIZ LOGIC ---
    function triggerQuiz(coinIndex) {
        if(isPaused) return;
        isPaused = true;
        currentPendingCoinIndex = coinIndex;
        
        // Stop player movement
        player.vx = 0;
        player.vy = 0;
        keys.right = false; // Reset keys to prevent drift
        keys.left = false;

        const qData = mathQuestions[coins[coinIndex].qIndex];
        
        quizQuestionEl.innerText = `Soal ${score + 1}: ${qData.q}`;
        quizOptionsEl.innerHTML = '';
        quizFeedbackEl.style.display = 'none';

        qData.options.forEach((opt, idx) => {
            const btn = document.createElement('div');
            btn.className = 'quiz-option';
            btn.innerText = opt;
            btn.onclick = () => checkAnswer(idx, qData.ans);
            quizOptionsEl.appendChild(btn);
        });

        quizModal.classList.remove('hidden');
    }

    function checkAnswer(selectedIdx, correctIdx) {
        if (selectedIdx === correctIdx) {
            // Correct
            quizModal.classList.add('hidden');
            isPaused = false;
            
            // Collect coin
            const c = coins[currentPendingCoinIndex];
            c.collected = true;
            score++;
            scoreEl.innerText = score;

            // Effects
            for(let k=0; k<20; k++) {
                particles.push(new Particle(c.x + c.size/2, c.y + c.size/2, '#ff0', 8));
            }
        } else {
            // Wrong
            quizFeedbackEl.style.display = 'block';
            // Shake effect or just text
            quizModal.style.borderColor = '#f00';
            setTimeout(() => quizModal.style.borderColor = '#0ff', 500);
        }
    }

    function startGame() {
        startScreen.classList.add('hidden');
        gameOverScreen.classList.add('hidden');
        initLevel();
        gameRunning = true;
        isPaused = false;
        deaths = 0;
        deathsEl.innerText = "0";
        requestAnimationFrame(gameLoop);
    }

    function resetLevel() {
        startGame();
    }

    function winGame() {
        gameRunning = false;
        gameOverScreen.classList.remove('hidden');
        goTitle.innerText = "SELAMAT!";
        goMsg.innerText = `Kamu lulus tes matematika & menyelesaikan rintangan!\nSkor: ${score}/6`;
    }

    function update() {
        if (!gameRunning) return;

        // Particles always update for visual flair even if paused
        for (let i = particles.length - 1; i >= 0; i--) {
            particles[i].update();
            if (particles[i].life <= 0) particles.splice(i, 1);
        }

        if (!isPaused) {
            player.update();
            // Camera follow
            camera.x = player.x - canvas.width / 3;
            if (camera.x < 0) camera.x = 0; // Don't scroll left of start
        }
    }

    function draw() {
        // Clear
        ctx.fillStyle = '#050505';
        ctx.fillRect(0, 0, canvas.width, canvas.height);

        // Grid Parallax
        ctx.strokeStyle = '#1a1a2e';
        ctx.lineWidth = 1;
        ctx.beginPath();
        let camX = Math.floor(camera.x);
        for(let x = -(camX % 50); x < canvas.width; x+=50) {
            ctx.moveTo(x, 0);
            ctx.lineTo(x, canvas.height);
        }
        for(let y = 0; y < canvas.height; y+=50) {
            ctx.moveTo(0, y);
            ctx.lineTo(canvas.width, y);
        }
        ctx.stroke();

        ctx.save();
        ctx.translate(-camera.x, -camera.y);

        // Draw Platforms
        for (let p of platforms) {
            if (p.type === 'normal') {
                ctx.fillStyle = '#000';
                ctx.strokeStyle = '#0ff';
                ctx.lineWidth = 2;
                ctx.shadowBlur = 10;
                ctx.shadowColor = '#0ff';
                ctx.fillRect(p.x, p.y, p.w, p.h);
                ctx.strokeRect(p.x, p.y, p.w, p.h);
            } else if (p.type === 'danger') {
                ctx.fillStyle = '#300';
                ctx.strokeStyle = '#f00';
                ctx.shadowBlur = 15;
                ctx.shadowColor = '#f00';
                ctx.fillRect(p.x, p.y, p.w, p.h);
                ctx.strokeRect(p.x, p.y, p.w, p.h);
                // Spikes
                ctx.beginPath();
                for(let i=p.x; i<p.x+p.w; i+=20) {
                    ctx.moveTo(i, p.y);
                    ctx.lineTo(i+10, p.y-10);
                    ctx.lineTo(i+20, p.y);
                }
                ctx.fillStyle = '#f00';
                ctx.fill();
            }
        }

        // Draw Portal
        ctx.shadowBlur = 30;
        ctx.shadowColor = '#f0f';
        ctx.fillStyle = '#200020';
        ctx.strokeStyle = '#f0f';
        ctx.lineWidth = 3;
        ctx.fillRect(portal.x, portal.y, portal.w, portal.h);
        ctx.strokeRect(portal.x, portal.y, portal.w, portal.h);
        ctx.fillStyle = `rgba(255, 0, 255, ${Math.random() * 0.5 + 0.2})`;
        ctx.beginPath();
        ctx.arc(portal.x + portal.w/2, portal.y + portal.h/2, 20, 0, Math.PI*2);
        ctx.fill();

        // Draw Coins (Only uncollected)
        for (let c of coins) {
            if (!c.collected) {
                ctx.shadowBlur = 15;
                ctx.shadowColor = '#ff0';
                ctx.fillStyle = '#ff0';
                ctx.beginPath();
                ctx.arc(c.x + c.size/2, c.y + c.size/2, c.size/2, 0, Math.PI * 2);
                ctx.fill();
                
                // Question Mark indicator
                ctx.fillStyle = '#000';
                ctx.font = 'bold 16px Courier';
                ctx.fillText("?", c.x + 6, c.y + 15);
            }
        }

        // Particles
        for (let p of particles) p.draw(ctx);

        // Player
        player.draw(ctx);

        ctx.restore();
    }

    function gameLoop(timestamp) {
        if (!lastTime) lastTime = timestamp;
        const deltaTime = timestamp - lastTime;
        lastTime = timestamp;

        update();
        draw();
        
        if (gameRunning) {
            requestAnimationFrame(gameLoop);
        }
    }

    function resize() {
        canvas.width = window.innerWidth;
        canvas.height = window.innerHeight;
    }
    window.addEventListener('resize', resize);
    resize();

    // Input Handling
    window.addEventListener('keydown', (e) => {
        if(isPaused) return; // Block input during quiz
        if (e.code === 'ArrowRight' || e.key === 'd') keys.right = true;
        if (e.code === 'ArrowLeft' || e.key === 'a') keys.left = true;
        if (e.code === 'ShiftLeft' || e.key === 'z') keys.dash = true;
        if ((e.code === 'Space' || e.code === 'ArrowUp' || e.key === 'w') && !keys.up) {
            keys.up = true;
            if (player) player.jump();
        }
    });

    window.addEventListener('keyup', (e) => {
        if (e.code === 'ArrowRight' || e.key === 'd') keys.right = false;
        if (e.code === 'ArrowLeft' || e.key === 'a') keys.left = false;
        if (e.code === 'ShiftLeft' || e.key === 'z') keys.dash = false;
        if (e.code === 'Space' || e.code === 'ArrowUp' || e.key === 'w') keys.up = false;
    });

    // Touch
    const btnLeft = document.getElementById('btn-left');
    const btnRight = document.getElementById('btn-right');
    const btnJump = document.getElementById('btn-jump');
    const btnDash = document.getElementById('btn-dash');

    const handleTouch = (el, key) => {
        el.addEventListener('touchstart', (e) => { 
            if(isPaused) return;
            e.preventDefault(); 
            keys[key] = true; 
            if(key === 'up' && player) player.jump(); 
        });
        el.addEventListener('touchend', (e) => { 
            e.preventDefault(); 
            keys[key] = false; 
        });
    };

    handleTouch(btnLeft, 'left');
    handleTouch(btnRight, 'right');
    handleTouch(btnJump, 'up');
    handleTouch(btnDash, 'dash');

    // Initial draw
    ctx.fillStyle = '#000';
    ctx.fillRect(0,0,canvas.width, canvas.height);
</script>
</body>
</html>


